use std::net::Ipv4Addr;
use std::sync::RwLock;

// Naive implementation. We have ~2**(32-prefix length) IP addresses available
// and bad things will happen if we end up with more entries. IPv6 solves this.
pub struct DnsLookupTable {
    base_addr: u32,
    entries: RwLock<Vec<String>>,
}

impl DnsLookupTable {
    pub fn new(base_address: Ipv4Addr) -> Self {
        let base_addr = u32::from(base_address);
        Self {
            base_addr,
            entries: RwLock::new(Vec::with_capacity(128)),
        }
    }

    pub fn get_hostname_for_ip(&self, ip: Ipv4Addr) -> Option<String> {
        let offset = (u32::from(ip) - self.base_addr) as usize;
        self.entries
            .read()
            .unwrap()
            .get(offset)
            .map(|s| s.to_owned())
    }

    pub fn get_or_insert(&self, hostname: &str) -> Ipv4Addr {
        // Use linear search for implementation simplicity.
        let pos = self
            .entries
            .read()
            .unwrap()
            .iter()
            .position(|e| e == hostname);

        let offset: u32 = pos
            .or_else(|| {
                let mut guard = self.entries.write().unwrap();
                let pos = guard.len();
                guard.push(hostname.to_owned());
                Some(pos)
            })
            .unwrap() as u32;
        Ipv4Addr::from(self.base_addr + offset)
    }
}
