use std::os::unix::io::AsRawFd;
use std::os::unix::io::FromRawFd;
use std::os::unix::prelude::OwnedFd;

// Trait to convert Unix error code into Result.
//
// Quick-and-dirty way to ensure error codes aren't ignored.
trait CErrToResult {
    fn ok(self) -> std::io::Result<libc::c_int>;
}

impl CErrToResult for libc::c_int {
    fn ok(self) -> std::io::Result<libc::c_int> {
        match self {
            err if err < 0 => Err(std::io::Error::last_os_error()),
            value => Ok(value),
        }
    }
}

// Abstraction to make it easier to convert a binary string into a fixed size
// array.
struct BStr(Vec<u8>);

impl BStr {
    fn from(value: &[u8]) -> Self {
        Self(Vec::from(value))
    }
}

impl<const N: usize> TryFrom<BStr> for [libc::c_char; N] {
    type Error = std::io::Error;

    fn try_from(value: BStr) -> Result<[libc::c_char; N], Self::Error> {
        if value.0.len() > N {
            return Err(Self::Error::new(
                std::io::ErrorKind::Other,
                format!(
                    "Content {:?} of length {} too long for array of length {}",
                    value.0,
                    value.0.len(),
                    N
                ),
            ));
        }
        let mut arr = [0 as libc::c_char; N];
        for (i, &c) in value.0.iter().enumerate() {
            arr[i] = c as libc::c_char;
        }
        Ok(arr)
    }
}

fn ifr_flags(flags: libc::c_short) -> libc::__c_anonymous_ifr_ifru {
    libc::__c_anonymous_ifr_ifru { ifru_flags: flags }
}

// Set loopback device as running.
//
// see `man 7 netdevice` for details.
fn set_lo_up() -> std::io::Result<i32> {
    let ifr = libc::ifreq {
        ifr_name: BStr::from(b"lo").try_into()?,
        ifr_ifru: ifr_flags(libc::IFF_UP as libc::c_short),
    };
    let sock_fd = {
        let sock_fd = unsafe { libc::socket(libc::AF_INET, libc::SOCK_DGRAM, 0) }.ok()?;
        unsafe { OwnedFd::from_raw_fd(sock_fd) }
    };

    unsafe { libc::ioctl(sock_fd.as_raw_fd(), libc::SIOCSIFFLAGS, &ifr) }.ok()
}

#[allow(unused)]
#[derive(Clone, Copy, PartialEq)]
#[repr(i32)]
pub enum CloneFlags {
    Files = libc::CLONE_FILES,
    Fs = libc::CLONE_FS,
    Newcgroup = libc::CLONE_NEWCGROUP,
    Newipc = libc::CLONE_NEWIPC,
    Newnet = libc::CLONE_NEWNET,
    Newns = libc::CLONE_NEWNS,
    Newpid = libc::CLONE_NEWPID,
    Newtime = libc::CLONE_NEWTIME,
    Newuser = libc::CLONE_NEWUSER,
    Newuts = libc::CLONE_NEWUTS,
    Svsem = libc::CLONE_SYSVSEM,
}

// #[derive(Clone, Copy)]
// pub struct UidGid {
//     uid: libc::uid_t,
//     gid: libc::gid_t,
// }

// impl UidGid {
//     pub fn from_current() -> Self {
//         let uid = unsafe { libc::getuid() };
//         let gid = unsafe { libc::getgid() };
//         Self { uid, gid }
//     }
// }

// Abstraction for unshare operations.
//
// Automatically brings lo up if a network namespace is created.
//
// See `man 2 unshare`.
pub struct Unshare {
    // flags: libc::c_int,
    // parent_uid_gid: UidGid,
}

impl Unshare {
    pub fn new(flags: &[CloneFlags]) -> std::io::Result<Self> {
        // let parent_uid_gid = UidGid::from_current();

        let mut flags_int = 0i32;
        for flag in flags {
            flags_int |= *flag as i32;
        }
        unsafe { libc::unshare(flags_int) }.ok()?;

        if flags.contains(&CloneFlags::Newnet) {
            set_lo_up()?;
        }
        Ok(Self {
            // flags: flags_int,
            // parent_uid_gid,
        })
    }
    // pub fn map_uid_gid(&self, uid_gid: UidGid) -> std::io::Result<()> {
    //     if self.flags & CloneFlags::Newuser as i32 == 0 {
    //         return Err(std::io::Error::new(
    //             std::io::ErrorKind::Other,
    //             "Can only map uid/gid for new user namespaces",
    //         ));
    //     }
    //     std::fs::write(
    //         "/proc/self/uid_map",
    //         format!("{} {} 1\n", uid_gid.uid, self.parent_uid_gid.uid),
    //     )?;
    //     std::fs::write("/proc/self/setgroups", "deny\n")?;
    //     std::fs::write(
    //         "/proc/self/gid_map",
    //         format!("{} {} 1\n", uid_gid.gid, self.parent_uid_gid.gid),
    //     )?;

    //     Ok(())
    // }
}
